package cn.ctyun.apis;

import cn.ctyun.sdk.*;
import com.fasterxml.jackson.annotation.JsonProperty;

import java.util.List;
import java.util.Map;

import static cn.ctyun.apis.Apis.ENDPOINT_NAME;

/** 调用该接口创建Kubernetes集群 */
public class CcseCreateClusterApi {
    private final CtyunRequestTemplate template;
    private final CtyunClient client;

    public CcseCreateClusterApi(CtyunClient client) {
        this.client = client;
        this.template =
                new CtyunRequestTemplate(
                        ENDPOINT_NAME, "POST", "/v2/cce/clusters/create", "application/json");
    }

    /**
     * 发起请求
     *
     * @param credential 用户信息
     * @param request 请求
     * @return 响应
     * @throws CtyunRequestException 异常
     */
    public CreateClusterResponse send(Credential credential, CreateClusterRequest request)
            throws CtyunRequestException {
        CtyunRequestBuilder builder = new CtyunRequestBuilder(template);
        builder.withCredential(credential);
        CtyunRequest ctReq = builder.build();
        if (request.regionId != null) {
            ctReq.addHeader("regionId", request.regionId);
            request.regionId = null;
        }
        ctReq.writeJson(request, template.contentType);
        CtyunResponse response = this.client.requestToEndpoint(ctReq);
        return response.parse(CreateClusterResponse.class);
    }

    public static class CreateClusterRequest {
        /**
         * 资源池ID，您可以查看<a href="https://www.ctyun.cn/document/10083472/11004422"
         * target="_blank">云容器引擎资源池</a> 另外您通过<a
         * href="https://www.ctyun.cn/document/10026730/10028695" target="_blank">地域和可用区</a>来了解资源池
         * 获取： <span style="background-color: rgb(73, 204, 144);color: rgb(255,255,255);padding:
         * 2px; margin:2px">查</span> <a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=25&api=5851&data=87&vid=81"
         * target="_blank">资源池列表查询</a>
         */
        @JsonProperty("regionId")
        private String regionId;

        /** 集群基本信息 */
        @JsonProperty("clusterBaseInfo")
        private CreateClusterClusterBaseInfoRequest clusterBaseInfo;

        /** master节点基本信息，专有版必填，托管版时不传 */
        @JsonProperty("masterHost")
        private CreateClusterMasterHostRequest masterHost;

        /** worker节点基本信息 */
        @JsonProperty("slaveHost")
        private CreateClusterSlaveHostRequest slaveHost;

        public String getRegionId() {
            return this.regionId;
        }

        public void setRegionId(String regionId) {
            this.regionId = regionId;
        }

        public CreateClusterClusterBaseInfoRequest getClusterBaseInfo() {
            return this.clusterBaseInfo;
        }

        public void setClusterBaseInfo(CreateClusterClusterBaseInfoRequest clusterBaseInfo) {
            this.clusterBaseInfo = clusterBaseInfo;
        }

        public CreateClusterMasterHostRequest getMasterHost() {
            return this.masterHost;
        }

        public void setMasterHost(CreateClusterMasterHostRequest masterHost) {
            this.masterHost = masterHost;
        }

        public CreateClusterSlaveHostRequest getSlaveHost() {
            return this.slaveHost;
        }

        public void setSlaveHost(CreateClusterSlaveHostRequest slaveHost) {
            this.slaveHost = slaveHost;
        }
    }

    public static class CreateClusterClusterBaseInfoRequest {
        /** 集群名字，不允许重复，名称长度4~40个字符，大小写字母开头，只能包含大小写字母、数字及分隔符(-)，大小写字母或数字结尾。 */
        @JsonProperty("clusterName")
        private String clusterName;

        /** 集群显示名称 */
        @JsonProperty("clusterAlias")
        private String clusterAlias;

        /**
         * 集群系列，cce.standard-专有版，cce.managed-托管版，您可查看<a
         * href="https://www.ctyun.cn/document/10083472/10892150">产品定义</a>选择
         */
        @JsonProperty("clusterSeries")
        private String clusterSeries;

        /**
         * 集群所属企业项目id，企业项目管理服务提供统一的云资源按企业项目管理，以及企业项目内的资源管理，成员管理。您可以通过查看<a
         * href="https://www.ctyun.cn/document/10017248/10017961">创建企业项目</a>了解如何创建企业项目 注：默认值为"0"
         */
        @JsonProperty("projectId")
        private String projectId;

        /** 计费模式：1为包周期，2为按需 */
        @JsonProperty("billMode")
        private String billMode;

        /** 订购周期类型：billMode=1时必填，3-月，5-1年，6-2年，7-3年 */
        @JsonProperty("cycleType")
        private String cycleType;

        /** 订购时长：billMode=1时必填，cycleType为3时，cycleCnt可选1-6；cycleType=5-7时，cycleCnt可选1 */
        @JsonProperty("cycleCnt")
        private String cycleCnt;

        /** 是否自动续订，默认false；按月购买，自动续订周期为1个月；按年购买，自动续订周期为1年。 */
        @JsonProperty("autoRenewStatus")
        private Boolean autoRenewStatus;

        /**
         * 集群版本1.23.3 ，1.25.6 ，1.27.8，1.29.3，1.31.6，您可查看<a
         * href="https://www.ctyun.cn/document/10083472/10650447">集群版本说明</a>选择。
         * 注：默认支持最高及以下的三个版本，其他版本需申请白名单，例如最高为1.31.6，则默认支持1.31.6，1.29.3，1.27.8
         */
        @JsonProperty("clusterVersion")
        private String clusterVersion;

        /** 托管版集群规格，托管版集群必填，单实例-managedbase，多实例-managedpro。单/多实例指控制面是否高可用，生产环境建议使用多实例 */
        @JsonProperty("seriesType")
        private String seriesType;

        /**
         * 托管版集群节点规模。当seriesType=managedbase时，选择节点规模，可选10；当seriesType=managedpro时，选择节点规模，可选为50，200，1000，2000
         */
        @JsonProperty("nodeScale")
        private String nodeScale;

        /**
         * ApiServer的ELB类型，standardI（标准I型） ,standardII（标准II型）, enhancedI（增强I型）, enhancedII（增强II型） ,
         * higherI（高阶I型） 您可查看不同<a
         * href="https://www.ctyun.cn/document/10026756/10032048">类型规格</a>的详细说明
         */
        @JsonProperty("elbProdCode")
        private String elbProdCode;

        /** 是否开启ApiServerEip，默认false，若开启将自动创建按需计费类型的eip。 */
        @JsonProperty("enableApiServerEip")
        private Boolean enableApiServerEip;

        /**
         * 虚拟私有云ID，vpc企业项目需跟集群企业项目相同。通过以下方式查询： <span style="background-color: rgb(73, 204,
         * 144);color: rgb(255,255,255);padding: 2px; margin:2px">查</span><a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=18&api=4814&data=94&vid=88">查询用户vpc列表</a>
         * 注：在多可用区类型资源池下，vpcID通常为“vpc-”开头，非多可用区类型资源池vpcID为uuid格式
         */
        @JsonProperty("vpcUuid")
        private String vpcUuid;

        /**
         * 子网id，需属于所选vpc。您可以查看<a
         * href="https://www.ctyun.cn/document/10026755/10098380">基本概念</a>来查找子网的相关定义 获取： <span
         * style="background-color: rgb(73, 204, 144);color: rgb(255,255,255);padding: 2px;
         * margin:2px">查</span><a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=18&amp;api=8659&amp;data=94">查询子网列表</a>
         * 注：在多可用区类型资源池下，subnetID通常以“subnet-”开头，非多可用区类型资源池subnetID为uuid格式
         */
        @JsonProperty("subnetUuid")
        private String subnetUuid;

        /** 是否开启nat网关，默认false，若开启将自动创建按需计费类型的nat网关。 */
        @JsonProperty("enableSnat")
        private Boolean enableSnat;

        /**
         * enableSnat=true必填，nat网关规格：small，medium，large，xlarge，您可查看不同<a
         * href="https://www.ctyun.cn/document/10026759/10043996">类型规格</a>的详细说明
         */
        @JsonProperty("natGatewaySpec")
        private String natGatewaySpec;

        /**
         * ip栈类型，ipv4或dual。dual表示双栈，默认ipv4，网络插件为calico时不支持ipv6。您可以通过查看<span style="background-color:
         * rgb(73, 204, 144);color: rgb(255,255,255);padding: 2px; margin:2px">查</span><a
         * href="https://www.ctyun.cn/document/10083472/10915742">搭建IPv4/IPv6双栈集群</a>了解详情
         */
        @JsonProperty("ipStackType")
        private String ipStackType;

        /** 是否自动创建安全组，默认false。 */
        @JsonProperty("autoGenerateSecurityGroup")
        private Boolean autoGenerateSecurityGroup;

        /**
         * 安全组id，需属于所选vpc，autoGenerateSecurityGroup=false必填，使用自定义安全组时，需要配置如下规则，参考<span
         * style="background-color: rgb(73, 204, 144);color: rgb(255,255,255);padding: 2px;
         * margin:2px">查</span><a
         * href="https://www.ctyun.cn/document/10083472/10915714">集群安全组规则配置</a> 安全组id您可以查看<a
         * href="https://www.ctyun.cn/document/10026755/10028520">安全组概述</a>了解安全组相关信息 获取： <span
         * style="background-color: rgb(73, 204, 144);color: rgb(255,255,255);padding: 2px;
         * margin:2px">查</span> <a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=18&api=4817&data=94&vid=88">查询用户安全组列表</a>
         * 注：在多可用区类型资源池下，安全组ID通常以“sg-”开头，非多可用区类型资源池安全组ID为uuid格式
         */
        @JsonProperty("securityGroupUuid")
        private String securityGroupUuid;

        /**
         * 网络插件，可选calico和cubecni。您可查看<a
         * href="https://www.ctyun.cn/document/10083472/10520760">容器网络插件说明</a>选择 注：calico需要申请白名单
         */
        @JsonProperty("networkPlugin")
        private String networkPlugin;

        /** 基于IPVLAN做弹性网卡共享，主机镜像只有使用CtyunOS系统才能生效 */
        @JsonProperty("ipvlan")
        private Boolean ipvlan;

        /** 提供基于策略的网络访问控制 */
        @JsonProperty("networkPolicy")
        private Boolean networkPolicy;

        /**
         * pod子网id列表，网络插件选择cubecni必传，需属于所选vpc。 获取： <span style="background-color: rgb(73, 204,
         * 144);color: rgb(255,255,255);padding: 2px; margin:2px">查</span><a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=18&amp;api=8659&amp;data=94">查询子网列表</a>
         */
        @JsonProperty("podSubnetUuidList")
        private List<String> podSubnetUuidList;

        /**
         * pod网络cidr，使用cubecni作为网络插件时，podCidr不传。calico网络插件时必传。
         * 注：网络插件为calico时，podCidr与vpcCidr与serviceCidr不能重叠。
         */
        @JsonProperty("podCidr")
        private String podCidr;

        /**
         * 服务cidr。网络插件为calico时，podCidr与vpcCidr与serviceCidr不能重叠。选择cubecni时，podCidr（vpcCidr）与serviceCidr不能重叠。
         */
        @JsonProperty("serviceCidr")
        private String serviceCidr;

        /** ipStackType=dual时必填，IPV6网段的掩码必须在112到120之间，如fc00::/112 */
        @JsonProperty("serviceCidrV6")
        private String serviceCidrV6;

        /**
         * kubeProxy类型：iptables或ipvs，您可查看<a
         * href="https://www.ctyun.cn/document/10083472/10915725">iptables与IPVS如何选择</a>
         */
        @JsonProperty("kubeProxy")
        private String kubeProxy;

        /** 节点服务开始端口，可选范围30000-65535 */
        @JsonProperty("startPort")
        private Long startPort;

        /** 节点服务终止端口，可选范围30000-65535，startPort到endPort范围需大于20 */
        @JsonProperty("endPort")
        private Long endPort;

        /**
         * 自定义证书SAN，您可查看<a
         * href="https://www.ctyun.cn/document/10083472/10915719">自定义集群APIServer证书SAN</a>
         */
        @JsonProperty("san")
        private String san;

        /** 是否开启集群CA，默认false */
        @JsonProperty("customCAEnable")
        private Boolean customCAEnable;

        /** 集群CA */
        @JsonProperty("customCA")
        private String customCA;

        /** 集群本地域名，集群创建完成后，暂无法修改，请慎重配置 */
        @JsonProperty("clusterDomain")
        private String clusterDomain;

        /** 集群删除保护 */
        @JsonProperty("deleteProtection")
        private Boolean deleteProtection;

        /** 集群标签 */
        @JsonProperty("clusterLabels")
        private Map<String,String> clusterLabels;

        /** 集群描述 */
        @JsonProperty("clusterDesc")
        private String clusterDesc;

        /** 服务账号令牌卷投影service-account-issuer，默认为https://kubernetes.default.svc */
        @JsonProperty("serviceAccountIssuer")
        private String serviceAccountIssuer;

        /** 服务账号令牌卷投影api-audiences，默认为https://kubernetes.default.svc */
        @JsonProperty("apiAudiences")
        private String apiAudiences;

        /** Kubelet自定义参数 */
        @JsonProperty("kubeletArgs")
        private CreateClusterClusterBaseInfoKubeletArgsRequest kubeletArgs;

        /** 时区 */
        @JsonProperty("timezone")
        private String timezone;

        /** 默认节点池名称，默认default */
        @JsonProperty("nodePool")
        private String nodePool;

        /** 可用区信息，包括可用区编码，该可用区下master节点数量。专有版时必填 */
        @JsonProperty("azInfos")
        private List<CreateClusterClusterBaseInfoAzInfosRequest> azInfos;

        /**
         * 容器运行时,可选containerd、docker，您可查看<a
         * href="https://www.ctyun.cn/document/10083472/10902208">容器运行时说明</a>
         */
        @JsonProperty("containerRuntime")
        private String containerRuntime;

        /** 主机登录方式，password：密码，secretPair：密钥对。默认为password */
        @JsonProperty("loginType")
        private String loginType;

        /**
         * 主机密码，loginType=password时必填，loginType不填时，password随机生成。满足以下规则： 物理机：用户密码，满足以下规则：长度在8～30个字符；
         * 必须包含大小写字母和（至少一个数字或者特殊字符）； 特殊符号可选：()`~!@#$%&*_-+=\ 云主机：长度在8～30个字符；
         * 必须包含大写字母、小写字母、数字以及特殊符号中的三项； 特殊符号可选：()`-!@#$%^&*_-+=｜{}[]:;'<>,.?/且不能以斜线号 / 开头；
         * 不能包含3个及以上连续字符；
         */
        @JsonProperty("password")
        private String password;

        /**
         * 密钥对id，loginType=secretPair时必填。您可以查看密钥对来了解密钥对相关内容 获取： <span style="background-color:
         * rgb(73, 204, 144);color: rgb(255,255,255);padding: 2px; margin:2px">查</span> <a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=25&api=8342&data=87&vid=81">查询一个或多个密钥对</a>
         * <span style="background-color: rgb(97, 175, 254);color: rgb(255,255,255);padding: 2px;
         * margin:2px">创</span> <a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=25&api=8344&data=87&vid=81"
         * target="_blank">创建一对SSH密钥对</a>
         */
        @JsonProperty("keyPairId")
        private String keyPairId;

        /**
         * 密钥对名称，loginType=secretPair时必填。您可以查看密钥对来了解密钥对相关内容 获取： <span style="background-color:
         * rgb(73, 204, 144);color: rgb(255,255,255);padding: 2px; margin:2px">查</span> <a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=25&api=8342&data=87&vid=81">查询一个或多个密钥对</a>
         * <span style="background-color: rgb(97, 175, 254);color: rgb(255,255,255);padding: 2px;
         * margin:2px">创</span> <a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=25&api=8344&data=87&vid=81"
         * target="_blank">创建一对SSH密钥对</a>
         */
        @JsonProperty("keyName")
        private String keyName;

        /** 自定义节点名称是否开启 */
        @JsonProperty("enableHostName")
        private Boolean enableHostName;

        /** 自定义节点名称前缀 */
        @JsonProperty("hostNamePrefix")
        private String hostNamePrefix;

        /** 自定义节点名称后缀 */
        @JsonProperty("hostNamePostfix")
        private String hostNamePostfix;

        /** 节点不可调度 */
        @JsonProperty("nodeUnschedulable")
        private Boolean nodeUnschedulable;

        /** 节点标签 */
        @JsonProperty("nodeLabels")
        private Map<String, String> nodeLabels;

        /**
         * 节点污点，格式为
         * [{\"key\":\"{key}\",\"value\":\"{value}\",\"effect\":\"{effect}\"}]，上述的{key}、{value}、{effect}替换为所需字段。effect枚举包括NoSchedule、PreferNoSchedule、NoExecute
         */
        @JsonProperty("nodeTaints")
        private String nodeTaints;

        /** 云主机资源标签 */
        @JsonProperty("resourceLabels")
        private Map<String, String> resourceLabels;

        /** 节点池节点标签的改动将会被同步更新到存量节点，默认false */
        @JsonProperty("syncNodeLabels")
        private Boolean syncNodeLabels;

        /** 节点池节点污点的改动将会被同步更新到存量节点，默认false */
        @JsonProperty("syncNodeTaints")
        private Boolean syncNodeTaints;

        /** 是否开启cpu管理策略，默认false */
        @JsonProperty("cpuManagerPolicyEnable")
        private Boolean cpuManagerPolicyEnable;

        /** 节点自定义脚本是否采用base64编码，默认为false */
        @JsonProperty("customScriptBase64")
        private Boolean customScriptBase64;

        /** 节点自定义脚本，如果customScriptBase64=false，此参数传明文；若customScriptBase64=true，此参数传base64后内容 */
        @JsonProperty("hostScript")
        private String hostScript;

        /** 节点部署后置脚本是否采用base64编码 */
        @JsonProperty("enablePostUserScript")
        private Boolean enablePostUserScript;

        /**
         * 节点部署后置脚本，节点自定义脚本，如果enablePostUserScript=false，此参数传明文；若enablePostUserScript=true，此参数传base64后内容
         */
        @JsonProperty("postUserScript")
        private String postUserScript;

        /**
         * 是否使用云主机组创建节点池，默认false；若enableAffinityGroup=false，表示不使用主机组；若enableAffinityGroup=true，affinityGroupUuid为空，表示使用默认反亲和主机组；若enableAffinityGroup=true，affinityGroupUuid不为空，表示使用已有主机组。该主机组配置仅对默认节点池生效，新建其他节点池可另外配置、
         */
        @JsonProperty("enableAffinityGroup")
        private Boolean enableAffinityGroup;

        /**
         * 是否使用已有反亲和性的云主机组创建节点池，enableAffinityGroup=true时填写，不填系统使用默认配置创建 获取： <span
         * style="background-color: rgb(73, 204, 144);color: rgb(255,255,255);padding: 2px;
         * margin:2px">查</span><a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=25&api=8324&data=87&vid=81">查询云主机组列表或者详情</a>
         */
        @JsonProperty("affinityGroupUuid")
        private String affinityGroupUuid;

        /** 工作节点使用已有委托，不填使用系统默认创建委托 */
        @JsonProperty("delegateName")
        private String delegateName;

        /** 节点池k8s节点名称类型，仅支持cceNodeName：与云主机保持一致 */
        @JsonProperty("k8sNodeNameType")
        private String k8sNodeNameType;

        /** 是否安装nginx ingress插件 */
        @JsonProperty("installNginxIngress")
        private Boolean installNginxIngress;

        /**
         * installNginxIngress=true必填，nginx ingress ELB的规格:standardI（标准I型） ,standardII（标准II型）,
         * enhancedI（增强I型）, enhancedII（增强II型） , higherI（高阶I型） 您可查看不同<a
         * href="https://www.ctyun.cn/document/10026756/10032048">规格的详细说明</a>
         */
        @JsonProperty("nginxIngressLBSpec")
        private String nginxIngressLBSpec;

        /**
         * installNginxIngress=true必填，nginx
         * ingress访问方式：external（公网），internal（内网），当选择公网时将自动创建eip额外产生eip相关费用
         */
        @JsonProperty("nginxIngressLBNetWork")
        private String nginxIngressLBNetWork;

        /** 是否安装NodeLocal DNSCache插件，默认为false */
        @JsonProperty("installDnsCache")
        private Boolean installDnsCache;

        /** 是否安装集群巡检插件，默认为false */
        @JsonProperty("installCubeEye")
        private Boolean installCubeEye;

        /**
         * 是否使用云日志服务。获得您的授权，以委托方式开通云日志服务，并自动为您安装 ctg-log-operator 插件以及创建 VPC 终端节点，以建立 CCE
         * 集群与云日志服务的连接通道。
         */
        @JsonProperty("enableLts")
        private Boolean enableLts;

        /**
         * 指定集群使用已有的云日志项目的项目名，enbaleLts需要设置为true，需与ltsProjectCode参数同时设置，获取：<span
         * style="background-color: rgb(73, 204, 144);color: rgb(255,255,255);padding: 2px;
         * margin:2px">查</span><a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=169&api=16658&data=216&isNormal=1&vid=295">云日志服务-项目分页列表</a>。若不指定则会默认创建一个集群专用的日志项目。
         */
        @JsonProperty("ltsProjectName")
        private String ltsProjectName;

        /**
         * 是否启用容器日志的采集，依赖云日志服务，enbaleLts需要设置为true。将创建容器日志的单元及采集规则，采集并上报集群所有命名空间的容器标准输出（cube-eventer、控制面pod、nginx-ingress-controller除外）到云日志服务（LTS）
         */
        @JsonProperty("enableContainerLog")
        private Boolean enableContainerLog;

        /**
         * 是否启用事件日志的采集，依赖云日志服务，enbaleLts需要设置为true。将自动为您安装cube-eventer插件，该插件是k8s事件采集插件，从集群apiserver获取k8s事件，并发送到对应的接收器中；并创建容器日志的单元及采集规则，采集并上报k8s事件到云日志服务（LTS）
         */
        @JsonProperty("enableEventLog")
        private Boolean enableEventLog;

        /** 是否启用控制面日志的采集，依赖云日志服务，enbaleLts需要设置为true。将创建集群控制面日志的单元及采集规则，采集并上报集群控制面日志到云日志服务（LTS） */
        @JsonProperty("enableControlPlaneLog")
        private Boolean enableControlPlaneLog;

        /**
         * 是否启用Kubernetes审计日志的采集，依赖云日志服务，enbaleLts需要设置为true。将创建Kubernetes审计日志的单元及采集规则，采集并上报Kubernetes审计日志到云日志服务（LTS）
         */
        @JsonProperty("enableAuditLog")
        private Boolean enableAuditLog;

        /**
         * 是否启用启用Ingress日志的采集，依赖云日志服务，enbaleLts需要设置为true。将创建Ingress日志的单元、采集规则及日志仪表盘，采集并上报Ingress日志到云日志服务（LTS）
         */
        @JsonProperty("enableIngressLog")
        private Boolean enableIngressLog;

        /** 节点池计费模式，对节点池内节点生效，1为包周期，2为按需 */
        @JsonProperty("nodePoolBillingMode")
        private String nodePoolBillingMode;

        /** 节点池计费周期，对节点池内节点生效，nodePoolBillingMode为1必传，3-月，5-1年，6-2年，7-3年 */
        @JsonProperty("nodePoolCycleType")
        private String nodePoolCycleType;

        /**
         * 节点池计费时长，对节点池内节点生效，nodePoolBillingMode=1时必填，nodePoolCycleType为3时，nodePoolCycleCount可选1-6；nodePoolCycleType=5-7时，nodePoolCycleCount可选1
         */
        @JsonProperty("nodePoolCycleCount")
        private Integer nodePoolCycleCount;

        /** 节点池是否支持自动续费，按月购买，自动续订周期为1个月；按年购买，自动续订周期为1年。 */
        @JsonProperty("nodePoolAutoRenewStatus")
        private Boolean nodePoolAutoRenewStatus;

        public String getClusterName() {
            return this.clusterName;
        }

        public void setClusterName(String clusterName) {
            this.clusterName = clusterName;
        }

        public String getClusterAlias() {
            return this.clusterAlias;
        }

        public void setClusterAlias(String clusterAlias) {
            this.clusterAlias = clusterAlias;
        }

        public String getClusterSeries() {
            return this.clusterSeries;
        }

        public void setClusterSeries(String clusterSeries) {
            this.clusterSeries = clusterSeries;
        }

        public String getProjectId() {
            return this.projectId;
        }

        public void setProjectId(String projectId) {
            this.projectId = projectId;
        }

        public String getBillMode() {
            return this.billMode;
        }

        public void setBillMode(String billMode) {
            this.billMode = billMode;
        }

        public String getCycleType() {
            return this.cycleType;
        }

        public void setCycleType(String cycleType) {
            this.cycleType = cycleType;
        }

        public String getCycleCnt() {
            return this.cycleCnt;
        }

        public void setCycleCnt(String cycleCnt) {
            this.cycleCnt = cycleCnt;
        }

        public Boolean getAutoRenewStatus() {
            return this.autoRenewStatus;
        }

        public void setAutoRenewStatus(Boolean autoRenewStatus) {
            this.autoRenewStatus = autoRenewStatus;
        }

        public String getClusterVersion() {
            return this.clusterVersion;
        }

        public void setClusterVersion(String clusterVersion) {
            this.clusterVersion = clusterVersion;
        }

        public String getSeriesType() {
            return this.seriesType;
        }

        public void setSeriesType(String seriesType) {
            this.seriesType = seriesType;
        }

        public String getNodeScale() {
            return this.nodeScale;
        }

        public void setNodeScale(String nodeScale) {
            this.nodeScale = nodeScale;
        }

        public String getElbProdCode() {
            return this.elbProdCode;
        }

        public void setElbProdCode(String elbProdCode) {
            this.elbProdCode = elbProdCode;
        }

        public Boolean getEnableApiServerEip() {
            return this.enableApiServerEip;
        }

        public void setEnableApiServerEip(Boolean enableApiServerEip) {
            this.enableApiServerEip = enableApiServerEip;
        }

        public String getVpcUuid() {
            return this.vpcUuid;
        }

        public void setVpcUuid(String vpcUuid) {
            this.vpcUuid = vpcUuid;
        }

        public String getSubnetUuid() {
            return this.subnetUuid;
        }

        public void setSubnetUuid(String subnetUuid) {
            this.subnetUuid = subnetUuid;
        }

        public Boolean getEnableSnat() {
            return this.enableSnat;
        }

        public void setEnableSnat(Boolean enableSnat) {
            this.enableSnat = enableSnat;
        }

        public String getNatGatewaySpec() {
            return this.natGatewaySpec;
        }

        public void setNatGatewaySpec(String natGatewaySpec) {
            this.natGatewaySpec = natGatewaySpec;
        }

        public String getIpStackType() {
            return this.ipStackType;
        }

        public void setIpStackType(String ipStackType) {
            this.ipStackType = ipStackType;
        }

        public Boolean getAutoGenerateSecurityGroup() {
            return this.autoGenerateSecurityGroup;
        }

        public void setAutoGenerateSecurityGroup(Boolean autoGenerateSecurityGroup) {
            this.autoGenerateSecurityGroup = autoGenerateSecurityGroup;
        }

        public String getSecurityGroupUuid() {
            return this.securityGroupUuid;
        }

        public void setSecurityGroupUuid(String securityGroupUuid) {
            this.securityGroupUuid = securityGroupUuid;
        }

        public String getNetworkPlugin() {
            return this.networkPlugin;
        }

        public void setNetworkPlugin(String networkPlugin) {
            this.networkPlugin = networkPlugin;
        }

        public Boolean getIpvlan() {
            return this.ipvlan;
        }

        public void setIpvlan(Boolean ipvlan) {
            this.ipvlan = ipvlan;
        }

        public Boolean getNetworkPolicy() {
            return this.networkPolicy;
        }

        public void setNetworkPolicy(Boolean networkPolicy) {
            this.networkPolicy = networkPolicy;
        }

        public List<String> getPodSubnetUuidList() {
            return this.podSubnetUuidList;
        }

        public void setPodSubnetUuidList(List<String> podSubnetUuidList) {
            this.podSubnetUuidList = podSubnetUuidList;
        }

        public String getPodCidr() {
            return this.podCidr;
        }

        public void setPodCidr(String podCidr) {
            this.podCidr = podCidr;
        }

        public String getServiceCidr() {
            return this.serviceCidr;
        }

        public void setServiceCidr(String serviceCidr) {
            this.serviceCidr = serviceCidr;
        }

        public String getServiceCidrV6() {
            return this.serviceCidrV6;
        }

        public void setServiceCidrV6(String serviceCidrV6) {
            this.serviceCidrV6 = serviceCidrV6;
        }

        public String getKubeProxy() {
            return this.kubeProxy;
        }

        public void setKubeProxy(String kubeProxy) {
            this.kubeProxy = kubeProxy;
        }

        public Long getStartPort() {
            return this.startPort;
        }

        public void setStartPort(Long startPort) {
            this.startPort = startPort;
        }

        public Long getEndPort() {
            return this.endPort;
        }

        public void setEndPort(Long endPort) {
            this.endPort = endPort;
        }

        public String getSan() {
            return this.san;
        }

        public void setSan(String san) {
            this.san = san;
        }

        public Boolean getCustomCAEnable() {
            return this.customCAEnable;
        }

        public void setCustomCAEnable(Boolean customCAEnable) {
            this.customCAEnable = customCAEnable;
        }

        public String getCustomCA() {
            return this.customCA;
        }

        public void setCustomCA(String customCA) {
            this.customCA = customCA;
        }

        public String getClusterDomain() {
            return this.clusterDomain;
        }

        public void setClusterDomain(String clusterDomain) {
            this.clusterDomain = clusterDomain;
        }

        public Boolean getDeleteProtection() {
            return this.deleteProtection;
        }

        public void setDeleteProtection(Boolean deleteProtection) {
            this.deleteProtection = deleteProtection;
        }

        public Map<String, String> getClusterLabels() {
            return this.clusterLabels;
        }

        public void setClusterLabels(Map<String, String> clusterLabels) {
            this.clusterLabels = clusterLabels;
        }

        public String getClusterDesc() {
            return this.clusterDesc;
        }

        public void setClusterDesc(String clusterDesc) {
            this.clusterDesc = clusterDesc;
        }

        public String getServiceAccountIssuer() {
            return this.serviceAccountIssuer;
        }

        public void setServiceAccountIssuer(String serviceAccountIssuer) {
            this.serviceAccountIssuer = serviceAccountIssuer;
        }

        public String getApiAudiences() {
            return this.apiAudiences;
        }

        public void setApiAudiences(String apiAudiences) {
            this.apiAudiences = apiAudiences;
        }

        public CreateClusterClusterBaseInfoKubeletArgsRequest getKubeletArgs() {
            return this.kubeletArgs;
        }

        public void setKubeletArgs(CreateClusterClusterBaseInfoKubeletArgsRequest kubeletArgs) {
            this.kubeletArgs = kubeletArgs;
        }

        public String getTimezone() {
            return this.timezone;
        }

        public void setTimezone(String timezone) {
            this.timezone = timezone;
        }

        public String getNodePool() {
            return this.nodePool;
        }

        public void setNodePool(String nodePool) {
            this.nodePool = nodePool;
        }

        public List<CreateClusterClusterBaseInfoAzInfosRequest> getAzInfos() {
            return this.azInfos;
        }

        public void setAzInfos(List<CreateClusterClusterBaseInfoAzInfosRequest> azInfos) {
            this.azInfos = azInfos;
        }

        public String getContainerRuntime() {
            return this.containerRuntime;
        }

        public void setContainerRuntime(String containerRuntime) {
            this.containerRuntime = containerRuntime;
        }

        public String getLoginType() {
            return this.loginType;
        }

        public void setLoginType(String loginType) {
            this.loginType = loginType;
        }

        public String getPassword() {
            return this.password;
        }

        public void setPassword(String password) {
            this.password = password;
        }

        public String getKeyPairId() {
            return this.keyPairId;
        }

        public void setKeyPairId(String keyPairId) {
            this.keyPairId = keyPairId;
        }

        public String getKeyName() {
            return this.keyName;
        }

        public void setKeyName(String keyName) {
            this.keyName = keyName;
        }

        public Boolean getEnableHostName() {
            return this.enableHostName;
        }

        public void setEnableHostName(Boolean enableHostName) {
            this.enableHostName = enableHostName;
        }

        public String getHostNamePrefix() {
            return this.hostNamePrefix;
        }

        public void setHostNamePrefix(String hostNamePrefix) {
            this.hostNamePrefix = hostNamePrefix;
        }

        public String getHostNamePostfix() {
            return this.hostNamePostfix;
        }

        public void setHostNamePostfix(String hostNamePostfix) {
            this.hostNamePostfix = hostNamePostfix;
        }

        public Boolean getNodeUnschedulable() {
            return this.nodeUnschedulable;
        }

        public void setNodeUnschedulable(Boolean nodeUnschedulable) {
            this.nodeUnschedulable = nodeUnschedulable;
        }

        public Map<String, String> getNodeLabels() {
            return this.nodeLabels;
        }

        public void setNodeLabels(Map<String, String> nodeLabels) {
            this.nodeLabels = nodeLabels;
        }

        public String getNodeTaints() {
            return this.nodeTaints;
        }

        public void setNodeTaints(String nodeTaints) {
            this.nodeTaints = nodeTaints;
        }

        public Map<String, String> getResourceLabels() {
            return this.resourceLabels;
        }

        public void setResourceLabels(Map<String, String> resourceLabels) {
            this.resourceLabels = resourceLabels;
        }

        public Boolean getSyncNodeLabels() {
            return this.syncNodeLabels;
        }

        public void setSyncNodeLabels(Boolean syncNodeLabels) {
            this.syncNodeLabels = syncNodeLabels;
        }

        public Boolean getSyncNodeTaints() {
            return this.syncNodeTaints;
        }

        public void setSyncNodeTaints(Boolean syncNodeTaints) {
            this.syncNodeTaints = syncNodeTaints;
        }

        public Boolean getCpuManagerPolicyEnable() {
            return this.cpuManagerPolicyEnable;
        }

        public void setCpuManagerPolicyEnable(Boolean cpuManagerPolicyEnable) {
            this.cpuManagerPolicyEnable = cpuManagerPolicyEnable;
        }

        public Boolean getCustomScriptBase64() {
            return this.customScriptBase64;
        }

        public void setCustomScriptBase64(Boolean customScriptBase64) {
            this.customScriptBase64 = customScriptBase64;
        }

        public String getHostScript() {
            return this.hostScript;
        }

        public void setHostScript(String hostScript) {
            this.hostScript = hostScript;
        }

        public Boolean getEnablePostUserScript() {
            return this.enablePostUserScript;
        }

        public void setEnablePostUserScript(Boolean enablePostUserScript) {
            this.enablePostUserScript = enablePostUserScript;
        }

        public String getPostUserScript() {
            return this.postUserScript;
        }

        public void setPostUserScript(String postUserScript) {
            this.postUserScript = postUserScript;
        }

        public Boolean getEnableAffinityGroup() {
            return this.enableAffinityGroup;
        }

        public void setEnableAffinityGroup(Boolean enableAffinityGroup) {
            this.enableAffinityGroup = enableAffinityGroup;
        }

        public String getAffinityGroupUuid() {
            return this.affinityGroupUuid;
        }

        public void setAffinityGroupUuid(String affinityGroupUuid) {
            this.affinityGroupUuid = affinityGroupUuid;
        }

        public String getDelegateName() {
            return this.delegateName;
        }

        public void setDelegateName(String delegateName) {
            this.delegateName = delegateName;
        }

        public String getK8sNodeNameType() {
            return this.k8sNodeNameType;
        }

        public void setK8sNodeNameType(String k8sNodeNameType) {
            this.k8sNodeNameType = k8sNodeNameType;
        }

        public Boolean getInstallNginxIngress() {
            return this.installNginxIngress;
        }

        public void setInstallNginxIngress(Boolean installNginxIngress) {
            this.installNginxIngress = installNginxIngress;
        }

        public String getNginxIngressLBSpec() {
            return this.nginxIngressLBSpec;
        }

        public void setNginxIngressLBSpec(String nginxIngressLBSpec) {
            this.nginxIngressLBSpec = nginxIngressLBSpec;
        }

        public String getNginxIngressLBNetWork() {
            return this.nginxIngressLBNetWork;
        }

        public void setNginxIngressLBNetWork(String nginxIngressLBNetWork) {
            this.nginxIngressLBNetWork = nginxIngressLBNetWork;
        }

        public Boolean getInstallDnsCache() {
            return this.installDnsCache;
        }

        public void setInstallDnsCache(Boolean installDnsCache) {
            this.installDnsCache = installDnsCache;
        }

        public Boolean getInstallCubeEye() {
            return this.installCubeEye;
        }

        public void setInstallCubeEye(Boolean installCubeEye) {
            this.installCubeEye = installCubeEye;
        }

        public Boolean getEnableLts() {
            return this.enableLts;
        }

        public void setEnableLts(Boolean enableLts) {
            this.enableLts = enableLts;
        }

        public String getLtsProjectName() {
            return this.ltsProjectName;
        }

        public void setLtsProjectName(String ltsProjectName) {
            this.ltsProjectName = ltsProjectName;
        }

        public Boolean getEnableContainerLog() {
            return this.enableContainerLog;
        }

        public void setEnableContainerLog(Boolean enableContainerLog) {
            this.enableContainerLog = enableContainerLog;
        }

        public Boolean getEnableEventLog() {
            return this.enableEventLog;
        }

        public void setEnableEventLog(Boolean enableEventLog) {
            this.enableEventLog = enableEventLog;
        }

        public Boolean getEnableControlPlaneLog() {
            return this.enableControlPlaneLog;
        }

        public void setEnableControlPlaneLog(Boolean enableControlPlaneLog) {
            this.enableControlPlaneLog = enableControlPlaneLog;
        }

        public Boolean getEnableAuditLog() {
            return this.enableAuditLog;
        }

        public void setEnableAuditLog(Boolean enableAuditLog) {
            this.enableAuditLog = enableAuditLog;
        }

        public Boolean getEnableIngressLog() {
            return this.enableIngressLog;
        }

        public void setEnableIngressLog(Boolean enableIngressLog) {
            this.enableIngressLog = enableIngressLog;
        }

        public String getNodePoolBillingMode() {
            return this.nodePoolBillingMode;
        }

        public void setNodePoolBillingMode(String nodePoolBillingMode) {
            this.nodePoolBillingMode = nodePoolBillingMode;
        }

        public String getNodePoolCycleType() {
            return this.nodePoolCycleType;
        }

        public void setNodePoolCycleType(String nodePoolCycleType) {
            this.nodePoolCycleType = nodePoolCycleType;
        }

        public Integer getNodePoolCycleCount() {
            return this.nodePoolCycleCount;
        }

        public void setNodePoolCycleCount(Integer nodePoolCycleCount) {
            this.nodePoolCycleCount = nodePoolCycleCount;
        }

        public Boolean getNodePoolAutoRenewStatus() {
            return this.nodePoolAutoRenewStatus;
        }

        public void setNodePoolAutoRenewStatus(Boolean nodePoolAutoRenewStatus) {
            this.nodePoolAutoRenewStatus = nodePoolAutoRenewStatus;
        }
    }

    public static class CreateClusterMasterHostRequest {
        /**
         * 实例规格名称，使用至少4C8G以上的规格，支持云主机 获取： <span style="background-color: rgb(73, 204, 144);color:
         * rgb(255,255,255);padding: 2px; margin:2px">查</span><a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=25&api=8327&data=87&isNormal=1&vid=81">查询主机规格资源</a>
         */
        @JsonProperty("itemDefName")
        private String itemDefName;

        /** 系统盘信息 */
        @JsonProperty("sysDisk")
        private CreateClusterMasterHostSysDiskRequest sysDisk;

        /** 数据盘信息 */
        @JsonProperty("dataDisks")
        private List<CreateClusterMasterHostDataDisksRequest> dataDisks;

        public String getItemDefName() {
            return this.itemDefName;
        }

        public void setItemDefName(String itemDefName) {
            this.itemDefName = itemDefName;
        }

        public CreateClusterMasterHostSysDiskRequest getSysDisk() {
            return this.sysDisk;
        }

        public void setSysDisk(CreateClusterMasterHostSysDiskRequest sysDisk) {
            this.sysDisk = sysDisk;
        }

        public List<CreateClusterMasterHostDataDisksRequest> getDataDisks() {
            return this.dataDisks;
        }

        public void setDataDisks(List<CreateClusterMasterHostDataDisksRequest> dataDisks) {
            this.dataDisks = dataDisks;
        }
    }

    public static class CreateClusterSlaveHostRequest {
        /**
         * 实例规格名称，使用至少4C8G以上的规格 获取： <span style="background-color: rgb(73, 204, 144);color:
         * rgb(255,255,255);padding: 2px; margin:2px">查</span><a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=25&api=8327&data=87&isNormal=1&vid=81">查询主机规格资源</a>
         */
        @JsonProperty("itemDefName")
        private String itemDefName;

        /** 系统盘信息，当itemDefName为裸金属规格时，需确认该裸金属是否支持系统盘云盘 */
        @JsonProperty("sysDisk")
        private CreateClusterSlaveHostSysDiskRequest sysDisk;

        /** 数据盘信息，当itemDefName为裸金属规格时，需确认该裸金属是否支持数据盘云盘 */
        @JsonProperty("dataDisks")
        private List<CreateClusterSlaveHostDataDisksRequest> dataDisks;

        /**
         * 镜像id，私有镜像必填，镜像CPU架构需跟规格匹配 获取： <span style="background-color: rgb(73, 204, 144);color:
         * rgb(255,255,255);padding: 2px; margin:2px">查</span><a
         * href="https://www.ctyun.cn/document/10083472/11004475">节点规格和节点镜像</a>
         */
        @JsonProperty("foreignMirrorId")
        private String foreignMirrorId;

        /**
         * 镜像类型，0-私有，1-公共。 您可查看<a href="https://www.ctyun.cn/document/10026730/10030151">镜像概述</a>
         */
        @JsonProperty("mirrorType")
        private Integer mirrorType;

        /**
         * 镜像名称，公共镜像必填，镜像CPU架构需跟规格匹配。您可查看<a
         * href="https://www.ctyun.cn/document/10083472/11004475">节点规格和节点镜像</a>
         */
        @JsonProperty("mirrorName")
        private String mirrorName;

        /** 可用区信息，包括可用区编码，可用区worker节点数量 */
        @JsonProperty("azInfos")
        private List<CreateClusterSlaveHostAzInfosRequest> azInfos;

        public String getItemDefName() {
            return this.itemDefName;
        }

        public void setItemDefName(String itemDefName) {
            this.itemDefName = itemDefName;
        }

        public CreateClusterSlaveHostSysDiskRequest getSysDisk() {
            return this.sysDisk;
        }

        public void setSysDisk(CreateClusterSlaveHostSysDiskRequest sysDisk) {
            this.sysDisk = sysDisk;
        }

        public List<CreateClusterSlaveHostDataDisksRequest> getDataDisks() {
            return this.dataDisks;
        }

        public void setDataDisks(List<CreateClusterSlaveHostDataDisksRequest> dataDisks) {
            this.dataDisks = dataDisks;
        }

        public String getForeignMirrorId() {
            return this.foreignMirrorId;
        }

        public void setForeignMirrorId(String foreignMirrorId) {
            this.foreignMirrorId = foreignMirrorId;
        }

        public Integer getMirrorType() {
            return this.mirrorType;
        }

        public void setMirrorType(Integer mirrorType) {
            this.mirrorType = mirrorType;
        }

        public String getMirrorName() {
            return this.mirrorName;
        }

        public void setMirrorName(String mirrorName) {
            this.mirrorName = mirrorName;
        }

        public List<CreateClusterSlaveHostAzInfosRequest> getAzInfos() {
            return this.azInfos;
        }

        public void setAzInfos(List<CreateClusterSlaveHostAzInfosRequest> azInfos) {
            this.azInfos = azInfos;
        }
    }

    public static class CreateClusterClusterBaseInfoClusterLabelsRequest {}

    public static class CreateClusterClusterBaseInfoKubeletArgsRequest {
        /** kubeAPIQPS */
        @JsonProperty("kubeAPIQPS")
        private Integer kubeAPIQPS;

        /** 请求至kube-apiserver的Burst配置 */
        @JsonProperty("kubeAPIBurst")
        private Integer kubeAPIBurst;

        /** kubelet管理的pod上限 */
        @JsonProperty("maxPods")
        private Integer maxPods;

        /** 每秒钟可以执行的镜像仓库拉取操作限值 */
        @JsonProperty("registryPullQPS")
        private Integer registryPullQPS;

        /** 突发性镜像拉取的上限值 */
        @JsonProperty("registryBurst")
        private Integer registryBurst;

        /** 限制Pod中的进程数 */
        @JsonProperty("podPidsLimit")
        private Integer podPidsLimit;

        /** 事件创建QPS限制 */
        @JsonProperty("eventRecordQPS")
        private Integer eventRecordQPS;

        /** 事件创建的个数的突发峰值上限 */
        @JsonProperty("eventBurst")
        private Integer eventBurst;

        /** 拓扑管理策略的资源对齐粒度 */
        @JsonProperty("topologyManagerScope")
        private String topologyManagerScope;

        /** 是否为设置了CPU限制的容器实施CPU CFS配额约束,默认值为true */
        @JsonProperty("cpuCFSQuota")
        private Boolean cpuCFSQuota;

        public Integer getKubeAPIQPS() {
            return this.kubeAPIQPS;
        }

        public void setKubeAPIQPS(Integer kubeAPIQPS) {
            this.kubeAPIQPS = kubeAPIQPS;
        }

        public Integer getKubeAPIBurst() {
            return this.kubeAPIBurst;
        }

        public void setKubeAPIBurst(Integer kubeAPIBurst) {
            this.kubeAPIBurst = kubeAPIBurst;
        }

        public Integer getMaxPods() {
            return this.maxPods;
        }

        public void setMaxPods(Integer maxPods) {
            this.maxPods = maxPods;
        }

        public Integer getRegistryPullQPS() {
            return this.registryPullQPS;
        }

        public void setRegistryPullQPS(Integer registryPullQPS) {
            this.registryPullQPS = registryPullQPS;
        }

        public Integer getRegistryBurst() {
            return this.registryBurst;
        }

        public void setRegistryBurst(Integer registryBurst) {
            this.registryBurst = registryBurst;
        }

        public Integer getPodPidsLimit() {
            return this.podPidsLimit;
        }

        public void setPodPidsLimit(Integer podPidsLimit) {
            this.podPidsLimit = podPidsLimit;
        }

        public Integer getEventRecordQPS() {
            return this.eventRecordQPS;
        }

        public void setEventRecordQPS(Integer eventRecordQPS) {
            this.eventRecordQPS = eventRecordQPS;
        }

        public Integer getEventBurst() {
            return this.eventBurst;
        }

        public void setEventBurst(Integer eventBurst) {
            this.eventBurst = eventBurst;
        }

        public String getTopologyManagerScope() {
            return this.topologyManagerScope;
        }

        public void setTopologyManagerScope(String topologyManagerScope) {
            this.topologyManagerScope = topologyManagerScope;
        }

        public Boolean getCpuCFSQuota() {
            return this.cpuCFSQuota;
        }

        public void setCpuCFSQuota(Boolean cpuCFSQuota) {
            this.cpuCFSQuota = cpuCFSQuota;
        }
    }

    public static class CreateClusterClusterBaseInfoAzInfosRequest {
        /**
         * 可用区编码查询： <span style="background-color: rgb(73, 204, 144);color:
         * rgb(255,255,255);padding: 2px; margin:2px">查</span><a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=25&api=5855&data=87&vid=81">资源池可用区查询</a>
         */
        @JsonProperty("azName")
        private String azName;

        /** 该可用区节点数量 */
        @JsonProperty("size")
        private Integer size;

        public String getAzName() {
            return this.azName;
        }

        public void setAzName(String azName) {
            this.azName = azName;
        }

        public Integer getSize() {
            return this.size;
        }

        public void setSize(Integer size) {
            this.size = size;
        }
    }

    public static class CreateClusterClusterBaseInfoNodeLabelsRequest {}

    public static class CreateClusterClusterBaseInfoResourceLabelsRequest {}

    public static class CreateClusterMasterHostSysDiskRequest {
        /**
         * 系统盘规格，云硬盘类型，取值范围： SATA：普通IO， SAS：高IO， SSD：超高IO 您可以查看<a
         * href="https://www.ctyun.cn/document/10027696/10162918">磁盘类型及性能介绍</a>磁盘类型相关信息
         */
        @JsonProperty("itemDefName")
        private String itemDefName;

        /** 系统盘大小，单位为G */
        @JsonProperty("size")
        private Integer size;

        public String getItemDefName() {
            return this.itemDefName;
        }

        public void setItemDefName(String itemDefName) {
            this.itemDefName = itemDefName;
        }

        public Integer getSize() {
            return this.size;
        }

        public void setSize(Integer size) {
            this.size = size;
        }
    }

    public static class CreateClusterMasterHostDataDisksRequest {
        /**
         * 数据盘规格名称，取值范围： SATA：普通IO， SAS：高IO， SSD：超高IO 您可以查看<a
         * href="https://www.ctyun.cn/document/10027696/10162918">磁盘类型及性能介绍</a>磁盘类型相关信息
         */
        @JsonProperty("itemDefName")
        private String itemDefName;

        /** 数据盘大小，单位为G。 系统盘： 1、裸金属：取值范围[100,2040] 2、云主机：取值范围[镜像磁盘大小,2040] 数据盘：取值范围[10,20000] */
        @JsonProperty("size")
        private Integer size;

        /**
         * 专属存储池id，需用专属资源池权限 获取： <span style="background-color: rgb(73, 204, 144);color:
         * rgb(255,255,255);padding: 2px; margin:2px">查</span><a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=4&api=12772&data=113&isNormal=1&vid=106">查询存储池信息详情</a>
         */
        @JsonProperty("decTypeId")
        private String decTypeId;

        public String getItemDefName() {
            return this.itemDefName;
        }

        public void setItemDefName(String itemDefName) {
            this.itemDefName = itemDefName;
        }

        public Integer getSize() {
            return this.size;
        }

        public void setSize(Integer size) {
            this.size = size;
        }

        public String getDecTypeId() {
            return this.decTypeId;
        }

        public void setDecTypeId(String decTypeId) {
            this.decTypeId = decTypeId;
        }
    }

    public static class CreateClusterSlaveHostSysDiskRequest {
        /**
         * 系统盘规格，云硬盘类型，取值范围： SATA：普通IO， SAS：高IO， SSD：超高IO 您可以查看<a
         * href="https://www.ctyun.cn/document/10027696/10162918">磁盘类型及性能介绍</a>磁盘类型相关信息
         */
        @JsonProperty("itemDefName")
        private String itemDefName;

        /** 系统盘大小，单位为G */
        @JsonProperty("size")
        private Integer size;

        public String getItemDefName() {
            return this.itemDefName;
        }

        public void setItemDefName(String itemDefName) {
            this.itemDefName = itemDefName;
        }

        public Integer getSize() {
            return this.size;
        }

        public void setSize(Integer size) {
            this.size = size;
        }
    }

    public static class CreateClusterSlaveHostDataDisksRequest {
        /**
         * 数据盘规格名称，取值范围： SATA：普通IO， SAS：高IO， SSD：超高IO 您可以查看<a
         * href="https://www.ctyun.cn/document/10027696/10162918">磁盘类型及性能介绍</a>磁盘类型相关信息
         */
        @JsonProperty("itemDefName")
        private String itemDefName;

        /** 数据盘大小，单位为G。 系统盘： 1、裸金属：取值范围[100,2040] 2、云主机：取值范围[镜像磁盘大小,2040] 数据盘：取值范围[10,20000] */
        @JsonProperty("size")
        private Integer size;

        /**
         * 专属存储池id，需用专属资源池权限 获取： <span style="background-color: rgb(73, 204, 144);color:
         * rgb(255,255,255);padding: 2px; margin:2px">查</span><a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=4&api=12772&data=113&isNormal=1&vid=106">查询存储池信息详情</a>
         */
        @JsonProperty("decTypeId")
        private String decTypeId;

        public String getItemDefName() {
            return this.itemDefName;
        }

        public void setItemDefName(String itemDefName) {
            this.itemDefName = itemDefName;
        }

        public Integer getSize() {
            return this.size;
        }

        public void setSize(Integer size) {
            this.size = size;
        }

        public String getDecTypeId() {
            return this.decTypeId;
        }

        public void setDecTypeId(String decTypeId) {
            this.decTypeId = decTypeId;
        }
    }

    public static class CreateClusterSlaveHostAzInfosRequest {
        /**
         * 可用区编码查询： <span style="background-color: rgb(73, 204, 144);color:
         * rgb(255,255,255);padding: 2px; margin:2px">查</span><a
         * href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=25&api=5855&data=87&vid=81">资源池可用区查询</a>
         */
        @JsonProperty("azName")
        private String azName;

        /** 该可用区节点数量 */
        @JsonProperty("size")
        private Integer size;

        public String getAzName() {
            return this.azName;
        }

        public void setAzName(String azName) {
            this.azName = azName;
        }

        public Integer getSize() {
            return this.size;
        }

        public void setSize(Integer size) {
            this.size = size;
        }
    }

    public static class CreateClusterResponse {
        /** 状态码 */
        @JsonProperty("statusCode")
        private Integer statusCode;

        /** 提示信息 */
        @JsonProperty("message")
        private String message;

        /** 错误码 */
        @JsonProperty("error")
        private String error;

        /** 返回对象 */
        @JsonProperty("returnObj")
        private CreateClusterReturnObjResponse returnObj;

        public Integer getStatusCode() {
            return this.statusCode;
        }

        public void setStatusCode(Integer statusCode) {
            this.statusCode = statusCode;
        }

        public String getMessage() {
            return this.message;
        }

        public void setMessage(String message) {
            this.message = message;
        }

        public String getError() {
            return this.error;
        }

        public void setError(String error) {
            this.error = error;
        }

        public CreateClusterReturnObjResponse getReturnObj() {
            return this.returnObj;
        }

        public void setReturnObj(CreateClusterReturnObjResponse returnObj) {
            this.returnObj = returnObj;
        }
    }

    public static class CreateClusterReturnObjResponse {
        /** 订单id */
        @JsonProperty("orderId")
        private String orderId;

        /** 订单编号 */
        @JsonProperty("orderNo")
        private String orderNo;

        public String getOrderId() {
            return this.orderId;
        }

        public void setOrderId(String orderId) {
            this.orderId = orderId;
        }

        public String getOrderNo() {
            return this.orderNo;
        }

        public void setOrderNo(String orderNo) {
            this.orderNo = orderNo;
        }
    }

    public static class CreateClusterTaintsRequest {
        /** 键 */
        @JsonProperty("key")
        private String key;

        /** 值 */
        @JsonProperty("value")
        private String value;

        /** 策略 */
        @JsonProperty("effect")
        private String effect;

        public String getKey() {
            return this.key;
        }

        public void setKey(String key) {
            this.key = key;
        }

        public String getValue() {
            return this.value;
        }

        public void setValue(String value) {
            this.value = value;
        }

        public String getEffect() {
            return this.effect;
        }

        public void setEffect(String effect) {
            this.effect = effect;
        }
    }
}
