#!/bin/bash

# 传入参数
# 对象存储地址 ip:port
url=$1
# 执行模式
# 0: 1停止zabbix并重命名
# 1: 1下包 2卸载旧telegraf 3安装新telegraf 4停止zabbix
mode=$2

# 设置初始化变量
# shell脚本路径
SHELL_FOLDER=$(
    cd "$(dirname "$0")" || exit
    pwd
)
# 对象存储桶路径
bucketPath="cttg"
# 二级路径
secondPath="linux"
# 对象存储中linux telegraf文件相应文件名
binFile="telegraf"
confFile="telegraf.conf"
execFile="exec.sh"
binSizeFile="size.txt"
# 下载超时时间 秒
timeout=30

# 判断系统架构
arch="amd64"
get_arch=$(arch)
if [[ ${get_arch} =~ "arm64" ]] || [[ ${get_arch} =~ "aarch64" ]]; then
    arch="arm64"
fi

# 判断init系统是否为systemd
# default: init_method="systemd" serviceFile="telegraf.service"
# systemd telegraf.service
# sysvinit init.sh
function find_init() {
    local method="systemd"
    local file="telegraf.service"

    # which返回不为0表示没有systemd
    which systemctl >/dev/null 2>&1
    if [ "$?" -ne 0 ]; then
        method="sysvinit"
        file="init.sh"
    fi
    echo "${method} ${file}"
}
  read -r init_method serviceFile <<<"$(find_init)"

# download $1:url $2:name
function download() {
    echo "downloading... path: $1 file: $2"
    curl --connect-timeout ${timeout} -o "$2" "$1"
}

# 下载最新telegraf相关文件
function download_files() {
    download "${url}/${bucketPath}/${secondPath}/${arch}/${binFile}" "${binFile}"
    download "${url}/${bucketPath}/${secondPath}/${confFile}" "${confFile}"
    download "${url}/${bucketPath}/${secondPath}/${execFile}" "${execFile}"
    download "${url}/${bucketPath}/${secondPath}/${serviceFile}" "${serviceFile}"
    download "${url}/${bucketPath}/${secondPath}/${arch}/${binSizeFile}" "${binSizeFile}"
}

#简单检查下载的telegraf bin文件是不是完整
function check_bin_file_size() {
    local binsize=`ls -l ./telegraf | awk -F" " '{print $5}'`
    local size=`cat ${binSizeFile}`
    if test "$binsize" = "$size"
    then
        echo "bin file size equal size:$size"
        return 0
    else
        echo "bin file size: ${binsize} not equal size:${size}"
        return 1
    fi

}

# 卸载旧的telegraf
function uninstall_telegraf() {
    if [[ ${init_method} = "systemd" ]]; then
        systemctl stop "${serviceFile}"
        pkill -9 "${binFile}"
        systemctl disable "${serviceFile}"

        # centos
        if [[ -d "/usr/lib/systemd/system" ]]; then
            rm -f "/usr/lib/systemd/system/${serviceFile}"
        # ubuntu
        elif [[ -d "/etc/systemd/system" ]]; then
            rm -f "/etc/systemd/system/${serviceFile}"
        fi
    else
        service "${binFile}" stop >/dev/null 2>&1
        pkill -9 "${binFile}"
        rm -f "/etc/init.d/${binFile}"
    fi

    rm -f "/usr/bin/${binFile}"
    #rm -rf "/etc/${binFile:?}"
    rm -rf "/etc/${binFile}/${confFile}"
    rm -rf "/etc/${binFile}/${execFile}"
    rm -rf "/var/log/${binFile:?}"
}

# 安装下载下来的telegraf
function install_telegraf() {
    cp "${binFile}" "/usr/bin/${binFile}"
    chmod +x "/usr/bin/${binFile}"
    if [ ! -d "/etc/${binFile}/${binFile}.d" ]; then
      mkdir -p "/etc/${binFile}/${binFile}.d"
    fi
    cp "${confFile}" "/etc/${binFile}/${confFile}"
    cp "${execFile}" "/etc/${binFile}/${execFile}"
    chmod +x "/etc/${binFile}/${execFile}"
    mkdir -p "/var/log/${binFile}"

    if [[ ${init_method} = "systemd" ]]; then
        # centos
        if [[ -d "/usr/lib/systemd/system" ]]; then
            cp "${serviceFile}" "/usr/lib/systemd/system/${serviceFile}"
        # ubuntu
        elif [[ -d "/etc/systemd/system" ]]; then
            cp "${serviceFile}" "/etc/systemd/system/${serviceFile}"
        fi

        systemctl daemon-reload
        systemctl enable "${serviceFile}"
        systemctl start "${serviceFile}"
    else
        cp "${serviceFile}" "/etc/init.d/${binFile}"
        chmod +x "/etc/init.d/${binFile}"

        chkconfig "${binFile}" on
        service "${binFile}" start
    fi
}

# 停止并重命名ctcm-agent服务
function stop_and_rename_zabbix() {
    if [[ ${init_method} = "systemd" ]]; then
        systemctl stop ctcm-agent.service
        pkill -9 ctcm_agentd
        systemctl disable ctcm-agent.service
        # centos
        if [[ -f "/usr/lib/systemd/system/ctcm-agent.service" ]]; then
            mv /usr/lib/systemd/system/ctcm-agent.service /usr/lib/systemd/system/ctcm-agent.service_deprecated
        # ubuntu
        elif [[ -f "/etc/systemd/system/ctcm-agent.service" ]]; then
            mv /etc/systemd/system/ctcm-agent.service /etc/systemd/system/ctcm-agent.service_deprecated
        fi
    else
        service ctcm-agent stop >/dev/null 2>&1
        pkill -9 ctcm_agentd
        if [[ -f "/etc/init.d/ctcm-agent" ]]; then
            mv /etc/init.d/ctcm-agent /etc/init.d/ctcm-agent_deprecated
        fi
    fi
    if [[ -f "/usr/sbin/ctcm_agentd" ]]; then
        mv /usr/sbin/ctcm_agentd /usr/sbin/ctcm_agentd_deprecated
    fi
}

function main() {
    echo "SHELL_FOLDER = ${SHELL_FOLDER}"
    echo "url = ${url:?"usage: sh $0 url <mode>"}"
    echo "mode = ${mode}"
    echo "bucketPath = ${bucketPath}"
    echo "secondPath = ${secondPath}"
    echo "binFile = ${binFile}"
    echo "confFile = ${confFile}"
    echo "execFile = ${execFile}"
    echo "binSizeFile = ${binSizeFile}"
    echo "timeout = ${timeout}"
    echo "arch = ${arch}"
    echo "init_method = ${init_method}"
    echo "serviceFile = ${serviceFile}"

    if [[ ${mode} == 1 ]]; then
        echo "download files start..."
        download_files
        echo "download files finished!"
        echo "check bin file size start..."
        check_bin_file_size
        RETVAL=$?
        if [ $RETVAL -ne 0 ]; then
             return 1
        fi
        echo "check bin file size finished!"
        echo "uninstall telegraf start..."
        uninstall_telegraf
        echo "uninstall telegraf finished!"
        echo "install telegraf start..."
        install_telegraf
        echo "install telegraf finished!"
    fi
    echo "stop and rename zabbix start..."
    stop_and_rename_zabbix
    echo "stop and rename zabbix finished!"

    echo "everthing is OK!"
}

main
