package apis

import (
	"context"
	"ctyun-go-sdk/core"
	"net/http"
	"strconv"
)

// CcseListClusterTasksApi
/* 查询任务详情
 */type CcseListClusterTasksApi struct {
	template core.CtyunRequestTemplate
	client   *core.CtyunClient
}

func NewCcseListClusterTasksApi(client *core.CtyunClient) *CcseListClusterTasksApi {
	return &CcseListClusterTasksApi{
		client: client,
		template: core.CtyunRequestTemplate{
			EndpointName: EndpointName,
			Method:       http.MethodGet,
			UrlPath:      "/v2/cce/tasks/{clusterId}/alltasks",
			ContentType:  "application/json",
		},
	}
}

func (a *CcseListClusterTasksApi) Do(ctx context.Context, credential core.Credential, req *CcseListClusterTasksRequest) (*CcseListClusterTasksResponse, error) {
	builder := core.NewCtyunRequestBuilder(a.template)
	builder = builder.ReplaceUrl("clusterId", req.ClusterId)
	builder.WithCredential(credential)
	ctReq := builder.Build()
	ctReq.AddHeader("regionId", req.RegionId)
	if req.PageNumber != 0 {
		ctReq.AddParam("pageNumber", strconv.FormatInt(int64(req.PageNumber), 10))
	}
	if req.PageSize != 0 {
		ctReq.AddParam("pageSize", strconv.FormatInt(int64(req.PageSize), 10))
	}
	response, err := a.client.RequestToEndpoint(ctx, ctReq)
	if err != nil {
		return nil, err
	}
	var resp CcseListClusterTasksResponse
	err = response.Parse(&resp)
	if err != nil {
		return nil, err
	}
	return &resp, nil
}

type CcseListClusterTasksRequest struct {
	ClusterId string `json:"clusterId,omitempty"` /*  集群ID，获取方式请参见<a href="https://www.ctyun.cn/document/10083472/11002105" target="_blank">如何获取接口URI中参数</a>。  */
	RegionId  string `json:"regionId,omitempty"`  /*  资源池ID，您可以查看<a href="https://www.ctyun.cn/document/10083472/11004422" target="_blank">云容器引擎资源池</a>
	另外您通过<a href="https://www.ctyun.cn/document/10026730/10028695" target="_blank">地域和可用区</a>来了解资源池
	获取：
	<span style="background-color: rgb(73, 204, 144);color: rgb(255,255,255);padding: 2px; margin:2px">查</span> <a href="https://eop.ctyun.cn/ebp/ctapiDocument/search?sid=25&api=5851&data=87&vid=81" target="_blank">资源池列表查询</a>  */
	PageNumber int32 `json:"pageNumber,omitempty"` /*  每页显示数量  */
	PageSize   int32 `json:"pageSize,omitempty"`   /*  分页查询页数  */
}

type CcseListClusterTasksResponse struct {
	StatusCode int32                                  `json:"statusCode"` /*  响应状态码  */
	RequestId  string                                 `json:"requestId"`  /*  请求ID  */
	Message    string                                 `json:"message"`    /*  响应信息  */
	ReturnObj  *CcseListClusterTasksReturnObjResponse `json:"returnObj"`  /*  请求结果  */
	Error      string                                 `json:"error"`      /*  错误码，参见错误码说明  */
}

type CcseListClusterTasksReturnObjResponse struct {
	Records []*CcseListClusterTasksReturnObjRecordsResponse `json:"records"` /*  诊断任务记录  */
	Total   int32                                           `json:"total"`   /*  总记录数  */
	Pages   int32                                           `json:"pages"`   /*  总页数  */
	Current int32                                           `json:"current"` /*  当前页数  */
	Size    int32                                           `json:"size"`    /*  每页大小  */
}

type CcseListClusterTasksReturnObjRecordsResponse struct {
	TaskId         string `json:"taskId"`         /*  任务ID  */
	ClusterId      string `json:"clusterId"`      /*  集群ID  */
	RegionId       string `json:"regionId"`       /*  资源池Id  */
	TaskType       string `json:"taskType"`       /*  任务类型  */
	TaskStatus     string `json:"taskStatus"`     /*  任务状态  */
	ParallelNumber int32  `json:"parallelNumber"` /*  并行数  */
	TaskContent    string `json:"taskContent"`    /*  任务内容  */
	TaskResult     string `json:"taskResult"`     /*  任务执行结果  */
	RetryTime      int32  `json:"retryTime"`      /*  重试次数  */
	CreatedTime    string `json:"createdTime"`    /*  创建时间  */
	ModifyTime     string `json:"modifyTime"`     /*  修改时间  */
}
